/*---------------------------------------------------------------------------*\
dynLocalAverageSmagorinsky - Implementation of the dynamic Smagorinsky
			     SGS model.
				 
Copyright Information
    Copyright (C) 1991-2009 OpenCFD Ltd.
    Copyright (C) 2010 Alberto Passalaua 

License
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::compressible::LESModels::dynSmagorinsky

Description
    The dynamic Smagorinsky model for compressible flows.
     From Subgrid-Scale Models for Compressible LES,Martin, Piomelli, & Candler
     Theoretical Comput. Fluid Dynamics 2000 13:361-376


SourceFiles
    dynSmagorinsky.C
    
Author
    Ayodeji Ojofeitimi, ayodejio@umd.edu

Notes
    Implementation of the dynamic Smagorinsky model with coefficients cD and
    cI computed as local average of their face values to avoid numerical 
    instabilities. 

    Negative values of the effective viscosity are removed by clipping it to
    zero (muSgs is clipped to -mu)

    The code is known to work with OpenFOAM 1.7.x

\*---------------------------------------------------------------------------*/

#ifndef dynSmagorinsky_H
#define dynSmagorinsky_H

#include "GenEddyVisc.H"
#include "LESfilter.H"
// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
namespace compressible
{
namespace LESModels
{

/*---------------------------------------------------------------------------*\
                           Class dynSmagorinsky Declaration
\*---------------------------------------------------------------------------*/

class dynSmagorinsky
:
    public GenEddyVisc
{
    // Private data

        volScalarField k_;

        autoPtr<LESfilter> filterPtr_;
        LESfilter& filter_;


    // Private Member Functions

        //- Update sub-grid scale fields
        void updateSubGridScaleFields(const volSymmTensorField& D);
	
        //- Calculate coefficients cD, cI from filtering velocity field
        volScalarField cD_(const volSymmTensorField& D) const;
        volScalarField cI_(const volSymmTensorField& D) const;
//        volScalarField cQ_(const volSymmTensorField& D) const;

        // Disallow default bitwise copy construct and assignment
        dynSmagorinsky(const dynSmagorinsky&);
        dynSmagorinsky& operator=(const dynSmagorinsky&);


public:

    //- Runtime type information
    TypeName("dynSmagorinsky");

    // Constructors

        //- Construct from components
        dynSmagorinsky
        (
            const volScalarField& rho,
            const volVectorField& U,
            const surfaceScalarField& phi,
            fluidThermo& thermoPhysicalModel,
            const word& turbulenceModelName = turbulenceModel::typeName,
            const word& modelName = typeName
        );

    //- Destructor
    virtual ~dynSmagorinsky()
    {}


    // Member Functions

        //- Return SGS kinetic energy
        tmp<volScalarField> k() const
        {
            return k_;
        }

        //- Correct Eddy-Viscosity and related properties
        virtual void correct(const tmp<volTensorField>& gradU);
	
        //- Read LESProperties dictionary
        virtual bool read();
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace LESModels
} // End namespace compressible
} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
