/*---------------------------------------------------------------------------*\
dynSmagorinsky - Implementation of the dynamic Smagorinsky
    			     SGS model.
    
Copyright Information
    Copyright (C) 1991-2009 OpenCFD Ltd.
    
License
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "dynSmagorinsky.H"
#include "addToRunTimeSelectionTable.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
namespace compressible
{
namespace LESModels
{

// * * * * * * * * * * * * * * Static Data Members * * * * * * * * * * * * * //

defineTypeNameAndDebug(dynSmagorinsky, 0);
addToRunTimeSelectionTable(LESModel, dynSmagorinsky, dictionary);

// * * * * * * * * * * * * * Private Member Functions  * * * * * * * * * * * //

void dynSmagorinsky::updateSubGridScaleFields(const volSymmTensorField& D)
{
    // The SGS viscosity is bounded so that muEff cannot become negative.
    // Values are limited here, and not in muEff, for consistency in stored
    // data and in submodels using muSgs().
    // No warning message is printed when this limitation is applied.
    muSgs_ = max(rho()*cD_(D)*sqr(delta())*sqrt(2.0*(D && D)), -mu());
    muSgs_.correctBoundaryConditions();
 
    alphaSgs_ = muSgs_/Prt_;
    alphaSgs_.correctBoundaryConditions();
}

volScalarField dynSmagorinsky::cD_(const volSymmTensorField& D) const
{
    volSymmTensorField Dfilter = filter_(rho()*D)/filter_(rho());
    volScalarField magSijfilter = sqrt(2.0*(Dfilter && Dfilter));
    volScalarField magSij = sqrt(2*(D && D));

    volSymmTensorField Lij = filter_(sqr(rho()*U())/rho()) - sqr(filter_(rho()*U()))/filter_(rho());
    volSymmTensorField Bij = -2*sqr(2*delta())*filter_(rho())*magSijfilter*dev(Dfilter);
    volSymmTensorField Aij = -2*sqr(delta())*rho()*magSij*dev(D);
    volSymmTensorField Mij = Bij - filter_(Aij);
    volScalarField LijMij = Lij && Mij;
    volScalarField MklMkl = Mij && Mij;
    
    return fvc::average(LijMij)/fvc::average(MklMkl);
}


volScalarField dynSmagorinsky::cI_(const volSymmTensorField& D) const
{
    volSymmTensorField Dfilter = filter_(rho()*D)/filter_(rho());
    volScalarField magSij = sqrt(2.0*(Dfilter && Dfilter));

    volSymmTensorField Lij = filter_(rho()*sqr(U())) - sqr(filter_(rho()*U()))/filter_(rho());
    volScalarField Lkk = tr(Lij);
    volScalarField beta = 2*sqr(2*delta())*filter_(rho())*sqr(magSij);
    volScalarField alpha = filter_(2*rho()*sqr(delta())*sqr(magSij));

    return (fvc::average(Lkk))/(fvc::average(beta - alpha));    
}

// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

dynSmagorinsky::dynSmagorinsky
(
    const volScalarField& rho,
    const volVectorField& U,
    const surfaceScalarField& phi,
    fluidThermo& thermoPhysicalModel,
    const word& turbulenceModelName,
    const word& modelName
)
:
    LESModel(modelName, rho, U, phi, thermoPhysicalModel, turbulenceModelName),
    GenEddyVisc(rho, U, phi, thermoPhysicalModel),

    k_
    (
        IOobject
        (
            "k",
            runTime_.timeName(),
            mesh_,
            IOobject::MUST_READ,
            IOobject::AUTO_WRITE
        ),
        mesh_
    )

    filterPtr_(LESfilter::New(U.mesh(), coeffDict())),
    filter_(filterPtr_())
{
    updateSubGridScaleFields(dev(symm(fvc::grad(U))));

    printCoeffs();
}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

void dynSmagorinsky::correct(const tmp<volTensorField>& tgradU)
{
    LESModel::correct(tgradU);
    const volTensorField& gradU = tgradU();
    GenEddyVisc::correct(gradU);
    volSymmTensorField D = symm(gradU);
    volScalarField magSij = sqrt(2.0*(D && D));
    k_ =2* cI_(D)*sqr(delta()*magSij);
    bound(k_,  kMin_);
    updateSubGridScaleFields(D);
}

bool dynSmagorinsky::read()
{
    if (GenEddyVisc::read())
    {
        filter_.read(coeffDict());

        return true;
    }
    else
    {
        return false;
    }
}


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace LESModels
} // End namespace compressible
} // End namespace Foam

// ************************************************************************* //
